<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Requests\CartRequest;
use App\Http\Requests\CheckoutRequest;
use App\Http\Requests\MultiCartRequest;
use App\Models\Cart;
use App\Repositories\CartRepository;
use App\Repositories\ProductRepository;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $isBuyNow = request()->is_buy_now ?? false;

        $carts = CartRepository::getCartItems($isBuyNow);

        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json('cart list', [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CartRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $product = ProductRepository::find($request->product_id);

        $quantity = $request->quantity ?? 1;

        $cart = CartRepository::getSingleCart($request->product_id, $isBuyNow);

        if ($isBuyNow) {
            CartRepository::removeBuyNowProduct($request, $product);
        }

        if (($product->quantity < $quantity) || ($product->quantity <= $cart?->quantity)) {
            return $this->json('Sorry! product cart quantity is limited. No more stock', [], 422);
        }

        // store or update cart
        CartRepository::storeOrUpdateByRequest($request, $product);

        $carts = CartRepository::getCartItems($isBuyNow);

        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json('product added to cart', [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function storeMultiple(MultiCartRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $token = request('access_token');
        $user = Auth::guard('sanctum')->user();

        if (! $token && ! $user) {
            return $this->json('Access token or user not found', [], 401);
        }

        $errors = [];

        foreach ($request->products ?? [] as $requestProduct) {

            $product = ProductRepository::find($requestProduct['product_id']);

            if (!$product) {
                $errors[] = "Product not found for ID: {$requestProduct['product_id']}";
                continue;
            }

            $quantity = array_key_exists('quantity', $requestProduct) ? $requestProduct['quantity'] ?? 1 : 1;
            $size = array_key_exists('size', $requestProduct) ? $requestProduct['size'] : null;
            $color = array_key_exists('color', $requestProduct) ? $requestProduct['color'] : null;
            $unit = array_key_exists('unit', $requestProduct) ? $requestProduct['unit'] : null;

            $cart = CartRepository::getSingleCart($product->id, $isBuyNow);

            if ($isBuyNow) {
                CartRepository::removeBuyNowProduct($request, $product);
            }

            if (($product->quantity < $quantity) || ($product->quantity <= $cart?->quantity)) {
                $errors[] = "Product ID {$product->id} has insufficient stock.";
                continue;
            }

            if ($cart) {
                $cart->update([
                    'quantity' => $isBuyNow ? 1 : $cart->quantity + 1,
                    'size' => $size ?? $cart->size,
                    'color' => $color ?? $cart->color,
                    'unit' => $unit ?? $cart->unit,
                ]);
            } else {
                Cart::create([
                    'access_token' => $token,
                    'product_id' => $product->id,
                    'shop_id' => $product->shop?->id,
                    'is_buy_now' => $isBuyNow,
                    'customer_id' => $user?->customer?->id ?? null,
                    'quantity' => $quantity ?? 1,
                    'size' => $size,
                    'color' => $color,
                    'unit' => $unit,
                ]);
            }
        }

        $carts = CartRepository::getCartItems($isBuyNow);

        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json(!empty($errors) ? $errors[0] : 'product added to cart', [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }

    /**
     * increase cart quantity
     */
    public function increment(CartRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $product = ProductRepository::find($request->product_id);

        $cart = CartRepository::getSingleCart($request->product_id, $isBuyNow);

        if (! $cart) {
            return $this->json('Sorry product not found in cart', [], 422);
        }

        $quantity = $cart->quantity;

        $flashSale = $product->flashSales?->first();

        $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

        $productQty = $product->quantity;

        if ($flashSaleProduct) {
            $flashSaleQty = $flashSaleProduct->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

            if ($flashSaleQty > 0) {
                $productQty = $flashSaleQty;
            }
        }

        if ($productQty > $quantity) {
            $cart->update([
                'quantity' => $quantity + 1,
            ]);
        } else {
            return $this->json('Sorry! product cart quantity is limited. No more stock', [], 422);
        }

        $carts = CartRepository::getCartItems($isBuyNow);
        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json('product quantity increased', [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }

    /**
     * decrease cart quantity
     * */
    public function decrement(CartRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $cart = CartRepository::getSingleCart($request->product_id, $isBuyNow);

        if (! $cart) {
            return $this->json('Sorry product not found in cart', [], 422);
        }

        $message = 'product removed from cart';

        if ($cart->quantity > 1) {
            $cart->update([
                'quantity' => $cart->quantity - 1,
            ]);

            $message = 'product quantity decreased';
        } else {
            $cart->delete();
        }

        $carts = CartRepository::getCartItems($isBuyNow);
        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json($message, [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }

    public function checkout(CheckoutRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $shopIds = $request->shop_ids ?? [];

        $carts = CartRepository::getCheckoutCartItems($isBuyNow, $shopIds);

        $checkout = CartRepository::checkoutByRequest($request, $carts);

        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        $message = 'Checkout information';

        $applyCoupon = false;

        if ($request->coupon_code && $checkout['coupon_discount'] > 0) {
            $applyCoupon = true;
            $message = 'Coupon applied';
        } elseif ($request->coupon_code) {
            $message = 'Coupon not applied';
        }

        return $this->json($message, [
            'checkout' => $checkout,
            'apply_coupon' => $applyCoupon,
            'checkout_items' => $shopWiseProducts,
        ]);
    }

    public function destroy(CartRequest $request)
    {
        $isBuyNow = $request->is_buy_now ?? false;

        $cart = Cart::where('product_id', $request->product_id)->first();

        if (! $cart) {
            return $this->json('Sorry product not found in cart', [], 422);
        }

        $cart->delete();

        $carts = CartRepository::getCartItems($isBuyNow);
        $groupCart = $carts->groupBy('shop_id');
        $shopWiseProducts = CartRepository::ShopWiseCartProducts($groupCart ?? []);

        return $this->json('product removed from cart', [
            'total' => $carts->count(),
            'cart_items' => $shopWiseProducts,
        ], 200);
    }
}
