<?php

namespace App\Http\Controllers\API;

use App\Enums\DeductionType;
use App\Enums\DiscountType;
use App\Enums\OrderStatus;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use App\Enums\Roles;
use App\Http\Controllers\Controller;
use App\Http\Requests\BuyNowRequest;
use App\Http\Requests\GuestBuyNowStoreRequest;
use App\Http\Requests\GuestOrderRequest;
use App\Http\Resources\UserResource;
use App\Models\AdminCoupon;
use App\Models\City;
use App\Models\Payment;
use App\Models\Product;
use App\Models\Shop;
use App\Models\User;
use App\Repositories\AddressRepository;
use App\Repositories\CouponRepository;
use App\Repositories\CustomerRepository;
use App\Repositories\OrderRepository;
use App\Repositories\ProductRepository;
use App\Repositories\ShopRepository;
use App\Repositories\UserRepository;
use App\Repositories\VatTaxRepository;
use App\Repositories\WalletRepository;
use Illuminate\Support\Collection;

class GuestOrderController extends Controller
{
    private ?User $user;

    private Collection $carts;

    private Collection $collectedCoupons;

    private Collection $products;

    private Collection $shops;

    public function store(GuestOrderRequest $request)
    {
        $this->carts = collect($request->carts);
        $carts = $this->carts->whereIn('shop_id', $request->shop_ids);
        $this->collectedCoupons = CouponRepository::query()
            ->Active()
            ->isValid()
            ->whereNull('limit_for_user')
            ->whereIn('id', $request->collected_coupons)
            ->get();

        if ($carts->isEmpty()) {
            return $this->json('Sorry shop cart is empty', [], 422);
        }

        $toUpper = strtoupper($request->payment_method);
        $paymentMethods = PaymentMethod::cases();
        $paymentMethod = $paymentMethods[array_search($toUpper, array_column(PaymentMethod::cases(), 'name'))];

        $sendToken = false;
        $this->user = UserRepository::query()->where('phone', $request->shipping['phone'])->first();

        if ($this->user) {
            if (! $this->user->customer) {
                return $this->json('Sorry, you can\'t place order with this phone number', [], 422);
            }

            if (! $this->user->password) {
                $sendToken = true;

                $this->user->update([
                    'name' => $request->shipping['name'],
                    'country' => $request->country,
                    'phone_code' => $request->phone_code,
                ]);
            }
        } else {
            $sendToken = true;

            // Create a new user
            $this->user = UserRepository::create([
                'name' => $request->shipping['name'],
                'phone' => $request->shipping['phone'],
                'country' => $request->shipping['country'],
                'phone_code' => $request->shipping['phone_code'],
                'is_active' => true,
            ]);

            // Create a new customer
            CustomerRepository::storeByRequest($this->user);

            // create wallet
            WalletRepository::storeByRequest($this->user);

            $this->user->assignRole(Roles::CUSTOMER->value);
        }

        $address = AddressRepository::create([
            'name' => $request->shipping['name'],
            'phone' => $request->shipping['phone'],
            'area' => $request->shipping['area'],
            'flat_no' => $request->shipping['flat_no'],
            'post_code' => $request->shipping['post_code'],
            'address_line' => $request->shipping['address_line'],
            'address_line2' => $request->shipping['address_line2'],
            'address_type' => 'home',
            'is_default' => false,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
        ]);

        $request->merge([
            'address_id' => $address->id,
        ]);

        $payment = $this->storeByrequestFromCart($request, $paymentMethod, $carts);

        $paymentUrl = null;
        if ($paymentMethod->name != 'CASH') {
            $paymentUrl = route('order.payment', ['payment' => $payment, 'gateway' => $request->payment_method]);
        }

        return $this->json('Order created successfully', [
            'order_payment_url' => $paymentUrl,
            'access' => $sendToken ? UserRepository::getAccessToken($this->user) : null,
            'user' => $sendToken ? new UserResource($this->user) : null,
        ]);
    }

    public function buyNow(BuyNowRequest $request)
    {
        $this->collectedCoupons = CouponRepository::query()
            ->Active()
            ->isValid()
            ->whereNull('limit_for_user')
            ->whereIn('id', $request->collected_coupons)
            ->get();

        $product = Product::find($request->product_id);
        $quantity = $request->quantity ?? 1;

        if ($product->quantity < $quantity) {
            return $this->json('Sorry, your product quantity out of stock', [], 422);
        }

        $price = $product->discount_price > 0 ? $product->discount_price : $product->price;

        $flashSale = $product->flashSales?->first();
        $flashSaleProduct = null;
        $flashQty = 0;

        if ($flashSale) {
            $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

            $flashQty = $flashSaleProduct?->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

            if ($flashQty == 0) {
                $flashSaleProduct = null;
            } else {
                $price = $flashSaleProduct->pivot->price;
            }
        }

        // calculate vat taxes
        $priceTaxAmount = 0;
        foreach ($product->vatTaxes ?? [] as $tax) {
            if ($tax->percentage > 0) {
                $priceTaxAmount += ($price * ($tax->percentage / 100));
            }
        }
        $price += $priceTaxAmount;

        $totalAmount = $price * $quantity;
        $shop = $product->shop;

        $sizePrice = $product->sizes()?->where('id', $request->size)->first()?->pivot?->price ?? 0;
        $totalAmount = $totalAmount + $sizePrice;

        $colorPrice = $product->colors()?->where('id', $request->color)->first()?->pivot?->price ?? 0;
        $totalAmount = $totalAmount + $colorPrice;

        $couponDiscount = $this->getCouponDiscount($totalAmount, $shop->id, $request->coupon_code);

        // get order base tax
        $totalOrderTaxAmount = 0;
        $orderBaseTax = VatTaxRepository::getOrderBaseTax();
        if ($orderBaseTax && $orderBaseTax->deduction == DeductionType::EXCLUSIVE->value && $orderBaseTax->percentage > 0) {
            $vatTaxAmount = $totalAmount * ($orderBaseTax->percentage / 100);
            $totalOrderTaxAmount += $vatTaxAmount;
        }

        $city = City::find($request->city_id) ?? null;
        $generalSetting = generaleSetting();

        // get delivery charge
        $deliveryCharge = $generalSetting?->default_delivery_charge ?? 0;
        if ($city) {
            $deliveryCharge = $city->delivery_charge;
            if ($city->amount > 0 && $totalAmount >= $city->amount) {
                $deliveryCharge = 0;
            }
        }

        $totalPayableAmount = ($totalAmount + $deliveryCharge + $totalOrderTaxAmount) - $couponDiscount['total_discount_amount'];

        $message = 'buy now order summary';

        $applyCoupon = false;

        if ($request->coupon_code && $couponDiscount['total_discount_amount'] > 0) {
            $applyCoupon = true;
            $message = 'Coupon applied';
        } elseif ($request->coupon_code) {
            $message = 'Coupon not applied';
        }

        return $this->json($message, [
            'total_amount' => (float) round($totalAmount, 2),
            'delivery_charge' => $deliveryCharge,
            'total_payable_amount' => (float) round($totalPayableAmount, 2),
            'order_tax_amount' => (float) round($totalOrderTaxAmount, 2),
            'apply_coupon' => $applyCoupon,
            'coupon_discount' => $couponDiscount['total_discount_amount'],
        ]);
    }

    public function buyNowStore(GuestBuyNowStoreRequest $request)
    {
        $this->collectedCoupons = CouponRepository::query()
            ->Active()
            ->isValid()
            ->whereNull('limit_for_user')
            ->whereIn('id', $request->collected_coupons)
            ->get();

        $product = Product::find($request->product_id);

        $quantity = $request->quantity ?? 1;

        if ($product->quantity < $quantity) {
            return $this->json('Sorry, your product quantity out of stock', [], 422);
        }

        $toUpper = strtoupper($request->payment_method);
        $paymentMethods = PaymentMethod::cases();
        $paymentMethod = $paymentMethods[array_search($toUpper, array_column(PaymentMethod::cases(), 'name'))];

        $sendToken = false;
        $this->user = UserRepository::query()->where('phone', $request->shipping['phone'])->first();

        if ($this->user) {
            if (! $this->user->customer) {
                return $this->json('Sorry, you can\'t place order with this phone number', [], 422);
            }

            if (! $this->user->password) {
                $sendToken = true;

                $this->user->update([
                    'name' => $request->shipping['name'],
                    'country' => $request->country,
                    'phone_code' => $request->phone_code,
                ]);
            }
        } else {
            $sendToken = true;

            // Create a new user
            $this->user = UserRepository::create([
                'name' => $request->shipping['name'],
                'phone' => $request->shipping['phone'],
                'country' => $request->shipping['country'],
                'phone_code' => $request->shipping['phone_code'],
                'is_active' => true,
            ]);

            // Create a new customer
            CustomerRepository::storeByRequest($this->user);

            // create wallet
            WalletRepository::storeByRequest($this->user);

            $this->user->assignRole(Roles::CUSTOMER->value);
        }

        $address = AddressRepository::create([
            'name' => $request->shipping['name'],
            'phone' => $request->shipping['phone'],
            'area' => $request->shipping['area'],
            'flat_no' => $request->shipping['flat_no'],
            'post_code' => $request->shipping['post_code'],
            'address_line' => $request->shipping['address_line'],
            'address_line2' => $request->shipping['address_line2'],
            'address_type' => 'home',
            'is_default' => false,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
        ]);

        $request->merge([
            'address_id' => $address->id,
        ]);

        $payment = $this->buyNowFromRequest($request, $product, $paymentMethod);

        $paymentUrl = null;
        if ($paymentMethod->name != 'CASH') {
            $paymentUrl = route('order.payment', ['payment' => $payment, 'gateway' => $request->payment_method]);
        }

        return $this->json('Order created successfully', [
            'order_payment_url' => $paymentUrl,
            'access' => $sendToken ? UserRepository::getAccessToken($this->user) : null,
            'user' => $sendToken ? new UserResource($this->user) : null,
        ]);
    }

    private function storeByrequestFromCart(GuestOrderRequest $request, $paymentMethod, $carts): Payment
    {
        $totalPayableAmount = 0;

        $payment = Payment::create([
            'amount' => $totalPayableAmount,
            'payment_method' => $request->payment_method,
        ]);

        $this->products = ProductRepository::query()->whereIn('id', $carts->pluck('product_id'))->get();
        $shopProducts = $carts->groupBy('shop_id');
        $this->shops = ShopRepository::query()->whereIn('id', $shopProducts->keys())->get();

        foreach ($shopProducts as $shopId => $cartProducts) {

            $shop = $this->shops->where('id', $shopId)->first();

            $cityId = $request->shipping['city_id'];
            $getCartAmounts = $this->getCartWiseAmounts($shop, collect($cartProducts), $request->coupon_code, $cityId);

            $lastOrderId = OrderRepository::query()->max('id');

            $order = OrderRepository::create([
                'shop_id' => $shop->id,
                'order_code' => str_pad($lastOrderId + 1, 6, '0', STR_PAD_LEFT),
                'prefix' => $shop->prefix ?? 'RC',
                'customer_id' => $this->user->customer->id,
                'coupon_id' => $getCartAmounts['coupon'],
                'delivery_charge' => $getCartAmounts['deliveryCharge'],
                'payable_amount' => $getCartAmounts['payableAmount'],
                'total_amount' => $getCartAmounts['totalAmount'],
                'tax_amount' => $getCartAmounts['taxAmount'],
                'coupon_discount' => $getCartAmounts['discount'],
                'payment_method' => $paymentMethod->value,
                'order_status' => OrderStatus::PENDING->value,
                'address_id' => $request->address_id,
                'instruction' => $request->note,
                'payment_status' => PaymentStatus::PENDING->value,
            ]);

            $totalPayableAmount += $getCartAmounts['payableAmount'];
            $payment->orders()->attach($order->id);

            foreach ($cartProducts as $cart) {
                $product = $this->products->where('id', $cart['product_id'])->first();
                $product->decrement('quantity', $cart['quantity']);
                $price = $product->discount_price > 0 ? $product->discount_price : $product->price;

                $flashSale = $product->flashSales?->first();
                $flashSaleProduct = null;
                $quantity = 0;

                $saleQty = $cart['quantity'];

                if ($flashSale) {
                    $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

                    $quantity = $flashSaleProduct?->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

                    if ($quantity == 0) {
                        $flashSaleProduct = null;
                    } else {
                        $price = $flashSaleProduct->pivot->price;
                        $saleQty += $flashSaleProduct->pivot->sale_quantity;

                        $flashSale->products()->updateExistingPivot($product->id, [
                            'sale_quantity' => $saleQty,
                        ]);
                    }
                }

                $order->products()->attach($cart['product_id'], [
                    'quantity' => $cart['quantity'],
                    'color' => $cart['color'],
                    'size' => $cart['size'],
                    'unit' => $cart['unit'],
                    'is_gift' => false,
                    'price' => $price,
                ]);
            }
        }

        $payment->update([
            'amount' => $totalPayableAmount,
        ]);

        return $payment;
    }

    private function buyNowFromRequest(GuestBuyNowStoreRequest $request, Product $product, $paymentMethod)
    {
        $quantity = $request->quantity ?? 1;

        $price = $product->discount_price > 0 ? $product->discount_price : $product->price;

        $flashSale = $product->flashSales?->first();
        $flashSaleProduct = null;
        $flashQty = 0;

        $saleQty = $quantity;

        if ($flashSale) {
            $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

            $flashQty = $flashSaleProduct?->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

            if ($flashQty == 0) {
                $flashSaleProduct = null;
            } else {
                $price = $flashSaleProduct->pivot->price;
                $saleQty += $flashSaleProduct->pivot->sale_quantity;

                $flashSale->products()->updateExistingPivot($product->id, [
                    'sale_quantity' => $saleQty,
                ]);
            }
        }

        // calculate vat taxes
        $priceTaxAmount = 0;
        foreach ($product->vatTaxes ?? [] as $tax) {
            if ($tax->percentage > 0) {
                $priceTaxAmount += ($price * ($tax->percentage / 100));
            }
        }
        $price += $priceTaxAmount;

        $totalAmount = $price * $quantity;

        $sizePrice = $product->sizes()?->where('id', $request->size)->first()?->pivot?->price ?? 0;
        $totalAmount += $sizePrice;

        $colorPrice = $product->colors()?->where('id', $request->color)->first()?->pivot?->price ?? 0;
        $totalAmount += $colorPrice;

        $couponDiscount = $this->getCouponDiscount($totalAmount, $product->shop->id, $request->coupon_code);

        $deliveryCharge = getDeliveryCharge($quantity);

        $shop = $product->shop;

        // get order base tax
        $totalOrderTaxAmount = 0;
        $orderBaseTax = VatTaxRepository::getOrderBaseTax();
        if ($orderBaseTax && $orderBaseTax->deduction == DeductionType::EXCLUSIVE->value && $orderBaseTax->percentage > 0) {
            $vatTaxAmount = $totalAmount * ($orderBaseTax->percentage / 100);
            $totalOrderTaxAmount += $vatTaxAmount;
        }

        $city = City::find($request->shipping['city_id']);
        $generalSetting = $generalSetting = generaleSetting();

        $deliveryCharge = $generalSetting?->default_delivery_charge ?? 0;
        if ($city) {
            $deliveryCharge = $city->delivery_charge;
            if ($city->amount > 0 && $totalAmount >= $city->amount) {
                $deliveryCharge = 0;
            }
        }

        $totalPayableAmount = ($totalAmount + $deliveryCharge + $totalOrderTaxAmount) - $couponDiscount['total_discount_amount'];

        $lastOrderId = OrderRepository::query()->max('id');

        $order = OrderRepository::create([
            'shop_id' => $shop->id,
            'order_code' => str_pad($lastOrderId + 1, 6, '0', STR_PAD_LEFT),
            'prefix' => $shop->prefix ?? 'RC',
            'customer_id' => $this->user->customer->id,
            'coupon_id' => $couponDiscount['coupon']?->id ?? null,
            'delivery_charge' => $deliveryCharge,
            'payable_amount' => $totalPayableAmount,
            'total_amount' => $totalAmount,
            'tax_amount' => $totalOrderTaxAmount,
            'coupon_discount' => $couponDiscount['total_discount_amount'],
            'payment_method' => $paymentMethod->value,
            'order_status' => OrderStatus::PENDING->value,
            'address_id' => $request->address_id,
            'instruction' => $request->note,
            'payment_status' => PaymentStatus::PENDING->value,
        ]);

        $product->decrement('quantity', $quantity);

        $size = $request->size ?? $product->sizes?->first()?->name;
        $color = $request->color ?? $product->colors?->first()?->name;
        $unit = $request->unit ?? $product->units?->first()?->name;

        $order->products()->attach($product->id, [
            'quantity' => $quantity,
            'color' => $color,
            'size' => $size,
            'unit' => $unit,
            'price' => $price,
        ]);

        $payment = null;
        if ($paymentMethod->value != PaymentMethod::CASH->value) {
            $payment = Payment::create([
                'amount' => $totalPayableAmount,
                'payment_method' => $request->payment_method,
            ]);
            $payment->orders()->attach($order->id);
        }

        return $payment;
    }

    private function getCartWiseAmounts(Shop $shop, $products, $couponCode = null, $cityId): array
    {
        $totalAmount = 0;
        $discount = 0;
        $coupon = null;
        $totalTaxAmount = 0;

        $city = City::find($cityId);
        $generalSetting = $generalSetting = generaleSetting();

        if (is_countable($products)) {
            foreach ($products as $cart) {
                $product = $this->products->where('id', $cart['product_id'])->first();
                $price = $product->discount_price > 0 ? $product->discount_price : $product->price;

                $flashSale = $product->flashSales?->first();
                $flashSaleProduct = null;
                $quantity = 0;

                if ($flashSale) {
                    $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

                    $quantity = $flashSaleProduct?->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

                    if ($quantity == 0) {
                        $flashSaleProduct = null;
                    } else {
                        $price = $flashSaleProduct->pivot->price;
                    }
                }

                $size = $product->sizes()?->where('id', $cart['size'])->first();
                $color = $product->colors()?->where('id', $cart['color'])->first();

                $sizePrice = $size?->pivot?->price ?? 0;
                $colorPrice = $color?->pivot?->price ?? 0;
                $extraPrice = $sizePrice + $colorPrice;

                $price = $price + $extraPrice;

                $taxAmount = 0;
                foreach ($product->vatTaxes ?? [] as $tax) {
                    if ($tax->percentage > 0) {
                        $taxAmount += $price * ($tax->percentage / 100);
                    }
                }
                $totalTaxAmount += $taxAmount;
                $price += $taxAmount;

                $totalAmount += ($price * $cart['quantity']);
            }
        } else {
            $product = $products->product;

            $price = $product->discount_price > 0 ? $product->discount_price : $product->price;

            $flashSale = $product->flashSales?->first();
            $flashSaleProduct = null;
            $quantity = 0;

            if ($flashSale) {
                $flashSaleProduct = $flashSale?->products()->where('id', $product->id)->first();

                $quantity = $flashSaleProduct?->pivot->quantity - $flashSaleProduct->pivot->sale_quantity;

                if ($quantity == 0) {
                    $flashSaleProduct = null;
                } else {
                    $price = $flashSaleProduct->pivot->price;
                }
            }

            $size = $product->sizes()?->where('id', $products?->size)->first();
            $color = $product->colors()?->where('id', $products?->color)->first();

            $sizePrice = $size?->pivot?->price ?? 0;
            $colorPrice = $color?->pivot?->price ?? 0;
            $extraPrice = $sizePrice + $colorPrice;

            $price = $price + $extraPrice;

            $taxAmount = 0;
            foreach ($product->vatTaxes ?? [] as $tax) {
                if ($tax->percentage > 0) {
                    $taxAmount += $price * ($tax->percentage / 100);
                }
            }
            $totalTaxAmount += $taxAmount;
            $price += $taxAmount;

            $totalAmount = $price * $products->quantity;
        }

        // get coupon discount
        $couponDiscount = $this->getCouponDiscount($totalAmount, $shop->id, $couponCode);

        // check coupon discount amount
        if ($couponDiscount['total_discount_amount'] > 0) {
            $discount += $couponDiscount['total_discount_amount'];
            $coupon = $couponDiscount['coupon'];
        }

        $deliveryCharge = $generalSetting?->default_delivery_charge ?? 0;
        if ($city) {
            $deliveryCharge = $city->delivery_charge;
            if ($city->amount > 0 && $totalAmount >= $city->amount) {
                $deliveryCharge = 0;
            }
        }

        // calculate payable amount
        $payableAmount = ($totalAmount + $deliveryCharge) - $discount;

        // return array
        return [
            'totalAmount' => $totalAmount,
            'payableAmount' => $payableAmount,
            'discount' => $discount,
            'deliveryCharge' => $deliveryCharge,
            'taxAmount' => $totalTaxAmount,
            'coupon' => $coupon?->id,
        ];
    }

    private function getCouponDiscount($totalAmount, $shopId, $couponCode = null)
    {
        $totalOrderAmount = 0;
        $totalDiscountAmount = 0;
        $coupon = null;

        if ($couponCode) {
            $shop = Shop::find($shopId);
            $coupon = $shop->coupons()->whereNull('limit_for_user')->where('code', $couponCode)->Active()->isValid()->first();

            if (! $coupon) {
                $coupon = AdminCoupon::where('shop_id', $shopId)->whereHas('coupon', function ($query) use ($couponCode) {
                    $query->whereNull('limit_for_user')->where('code', $couponCode)->Active()->isValid();
                })->first()?->coupon;
            }

            if ($coupon) {
                $discount = $this->getCouponDiscountAmount($coupon, $totalAmount);

                $totalOrderAmount += $discount['total_amount'];
                $totalDiscountAmount += $discount['discount_amount'];
            }
        } else {
            $collectedCoupons = $this->collectedCoupons->where('shop_id', $shopId);

            $adminCoupons = AdminCoupon::where('shop_id', $shopId)->whereHas('coupon', function ($query) {
                $query->whereNull('limit_for_user')->Active()->isValid();
            })->get();

            foreach ($adminCoupons as $adminCoupon) {
                if (in_array($adminCoupon->coupon_id, $this->collectedCoupons->pluck('id')->toArray())) {
                    $collectedCoupons->push($adminCoupon->coupon);
                }
            }

            foreach ($collectedCoupons as $collectedCoupon) {
                $discount = $this->getCouponDiscountAmount($collectedCoupon, $totalAmount);

                $totalOrderAmount += (float) $discount['total_amount'];

                if ($discount['discount_amount'] > 0) {
                    $coupon = $collectedCoupon;
                    $totalDiscountAmount += (float) $discount['discount_amount'];
                    break;
                }
            }
        }

        return [
            'total_order_amount' => $totalOrderAmount,
            'total_discount_amount' => $totalDiscountAmount,
            'coupon' => $coupon,
        ];
    }

    private function getCouponDiscountAmount($coupon, $totalAmount)
    {
        $amount = $coupon->type->value == DiscountType::PERCENTAGE->value ? ($totalAmount * $coupon->discount) / 100 : $coupon->discount;
        $couponDiscount = 0;

        if ($coupon->min_amount <= $totalAmount) {
            $couponDiscount = $amount;

            if ($coupon->max_discount_amount && $coupon->max_discount_amount < $amount) {
                $couponDiscount = $coupon->max_discount_amount;
            }
        }

        return [
            'total_amount' => $totalAmount,
            'discount_amount' => (float) round($couponDiscount ?? 0, 2),
        ];
    }
}
