<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Resources\LanguageResource;
use App\Http\Resources\PaymentGatewayResource;
use App\Http\Resources\SocialLinkResource;
use App\Models\Currency;
use App\Models\GeneraleSetting;
use App\Models\PaymentGateway;
use App\Models\SocialAuth;
use App\Models\SocialLink;
use App\Models\VerifyManage;
use App\Repositories\LanguageRepository;
use App\Repositories\ThemeColorRepository;
use Illuminate\Support\Facades\Cache;

class MasterController extends Controller
{
    /**
     * Get master data for all.
     *
     * @return mixed
     */
    public function index()
    {
        $generaleSetting = Cache::rememberForever('generale_setting', function () {
            return GeneraleSetting::first();
        });

        $paymentGateways = Cache::rememberForever('payment_gateway', function () {
            return PaymentGateway::where('is_active', true)->get();
        });

        $verifyManage = Cache::rememberForever('verify_manage', function () {
            return VerifyManage::first();
        });

        $shopType = $generaleSetting?->shop_type ?? 'multi';

        $socialLinks = SocialLink::whereNotNull('link')->get();
        // add whatsapp link
        if (! $socialLinks->pluck('name')->contains('WhatsApp') && $generaleSetting?->mobile) {
            $socialLinks->push((object) [
                'name' => 'whatsapp',
                'link' => 'https://api.whatsapp.com/send?phone='.$generaleSetting?->mobile,
                'logo' => '/assets/icons/whatsapp.svg',
            ]);
        }

        $themeColor = ThemeColorRepository::query()->where('is_default', true)->first();

        $themeColors = (object) [
            'primary' => $themeColor ? $themeColor['primary'] : '#ef6d25',
            'primary50' => $themeColor ? $themeColor['variant_50'] : '#fdf0e9',
            'primary100' => $themeColor ? $themeColor['variant_100'] : '#fce2d3',
            'primary200' => $themeColor ? $themeColor['variant_200'] : '#f9c5a8',
            'primary300' => $themeColor ? $themeColor['variant_300'] : '#f5a77c',
            'primary400' => $themeColor ? $themeColor['variant_400'] : '#f28a51',
            'primary500' => $themeColor ? $themeColor['variant_500'] : '#ef6d25',
            'primary600' => $themeColor ? $themeColor['variant_600'] : '#d76221',
            'primary700' => $themeColor ? $themeColor['variant_700'] : '#bf571e',
            'primary800' => $themeColor ? $themeColor['variant_800'] : '#a74c1a',
            'primary900' => $themeColor ? $themeColor['variant_900'] : '#8f4116',
            'primary950' => $themeColor ? $themeColor['variant_950'] : '#783713',
        ];

        $languages = Cache::remember('languages', 60 * 24 * 30, function () {
            return LanguageRepository::getAll();
        });

        $socialAuths = collect([]);
        foreach (SocialAuth::all() as $socialAuth) {
            $socialAuths[$socialAuth->provider] = [
                'name' => $socialAuth->name,
                'client_id' => $socialAuth->client_id,
                'is_active' => (bool) $socialAuth->is_active,
                'redirect_url' => $socialAuth->redirect,
            ];
        }

        $allCurrencies = Cache::rememberForever('currencies', function () {
            return Currency::where('is_active', true)->get();
        });

        // default currency rate
        $defaultCurrency = generaleSetting('defaultCurrency');

        $defaultCurrencyRate = $defaultCurrency?->rate ?? 1;

        $currencies = $allCurrencies->map(function ($currency) use ($defaultCurrencyRate) {
            $rateFromDefault = $currency->rate / $defaultCurrencyRate;

            return [
                'id' => $currency->id,
                'name' => $currency->name,
                'symbol' => $currency->symbol,
                'rate' => (float) $currency->rate,
                'rate_from_default' => round($rateFromDefault, 4),
                'is_default' => (bool) $currency->is_default,
            ];
        });

        return $this->json('Master data', [
            'currency' => [
                'name' => $defaultCurrency?->name ?? 'USD',
                'symbol' => $generaleSetting?->currency ?? '$',
                'rate' => (float) $defaultCurrencyRate,
                'position' => $generaleSetting?->currency_position ?? 'prefix',
            ],
            'currencies' => $currencies,
            'app_locale' => env('APP_LOCALE', 'en'),
            'app_name' => $generaleSetting?->name ?? config('app.name'),
            'show_download_app' => (bool) ($generaleSetting?->show_download_app ?? true),
            'google_playstore_link' => $generaleSetting?->google_playstore_url ?? null,
            'app_store_link' => $generaleSetting?->app_store_url ?? null,
            'payment_gateways' => PaymentGatewayResource::collection($paymentGateways),
            'multi_vendor' => (bool) ($shopType == 'multi' ? true : false),
            'mobile' => $generaleSetting?->footer_phone ?? '+88 01365 236 543',
            'email' => $generaleSetting?->footer_email ?? 'support@mayamart.com',
            'address' => $generaleSetting?->address ?? 'Dhaka, Bangladesh',
            'web_show_footer' => (bool) ($generaleSetting?->show_footer ?? true),
            'web_footer_text' => $generaleSetting?->footer_text ?? 'All right reserved by RazinSoft',
            'web_footer_description' => $generaleSetting?->footer_description ?? 'The ultimate all-in-one solution for your eCommerce business worldwide.',
            'web_logo' => $generaleSetting?->logo ?? asset('assets/logo.png'),
            'web_footer_logo' => $generaleSetting?->footerLogo ?? asset('assets/logoWhite.png'),
            'app_name' => $generaleSetting?->name ?? config('app.name'),
            'app_logo' => $generaleSetting?->appLogo ?? asset('assets/favicon.png'),
            'footer_qr' => $generaleSetting?->footerQr ?? null,
            'social_links' => SocialLinkResource::collection($socialLinks),
            'theme_colors' => $themeColors,
            'pusher_app_key' => config('broadcasting.connections.pusher.key'),
            'pusher_app_cluster' => config('broadcasting.connections.pusher.options.cluster'),
            'app_environment' => config('app.env'),
            'register_otp_verify' => (bool) ($verifyManage?->register_otp ?? false),
            'register_otp_type' => $verifyManage?->register_otp_type ?? 'phone',
            'forgot_otp_type' => $verifyManage?->forgot_otp_type ?? 'phone',
            'languages' => LanguageResource::collection($languages),
            'social_auths' => $socialAuths,
        ]);
    }
}
