<?php

namespace App\Http\Controllers\Gateway\PayPal;

use App\Enums\PaymentStatus;
use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\PaymentGateway;
use Illuminate\Http\Request;
use PaypalServerSdkLib\Authentication\ClientCredentialsAuthCredentialsBuilder;
use PaypalServerSdkLib\Environment;
use PaypalServerSdkLib\Models\Builders\AmountWithBreakdownBuilder;
use PaypalServerSdkLib\Models\Builders\OrderApplicationContextBuilder;
use PaypalServerSdkLib\Models\Builders\OrderRequestBuilder;
use PaypalServerSdkLib\Models\Builders\PurchaseUnitRequestBuilder;
use PaypalServerSdkLib\Models\CheckoutPaymentIntent;
use PaypalServerSdkLib\PaypalServerSdkClientBuilder;

class ProcessController extends Controller
{
    /**
     * Process to PayPal
     *
     * @return string
     */
    public static function process($paymentGateway, Payment $payment)
    {
        $config = json_decode($paymentGateway->config);

        $environment = $paymentGateway->mode === 'live' ? Environment::PRODUCTION : Environment::SANDBOX;

        $client = PaypalServerSdkClientBuilder::init()
            ->clientCredentialsAuthCredentials(
                ClientCredentialsAuthCredentialsBuilder::init(
                    $config->client_id,
                    $config->client_secret
                )
            )
            ->environment($environment)
            ->build();

        // default currency rate
        $defaultCurrency = generaleSetting('defaultCurrency');

        $defaultCurrencyRate = $defaultCurrency?->rate ?? 1;
        $rateFromDefault = 1 / $defaultCurrencyRate;

        $amount = round($payment->amount * $rateFromDefault, 2);

        $collect = [
            'body' => OrderRequestBuilder::init(
                CheckoutPaymentIntent::CAPTURE,
                [
                    PurchaseUnitRequestBuilder::init(
                        AmountWithBreakdownBuilder::init(
                            'USD',
                            $amount
                        )->build()
                    )->build()
                ]
            )
            ->applicationContext(
                OrderApplicationContextBuilder::init()
                    ->returnUrl(route('paypal.payment.success', $payment))
                    ->cancelUrl(route('order.payment.cancel', $payment))
                    ->build()
            )
            ->build(),
            'prefer' => 'return=minimal'
        ];

        $apiResponse = $client->getOrdersController()->createOrder($collect);

        if ($apiResponse->isSuccess()) {
            /** @var \PaypalServerSdkLib\Models\Order $order */
            $order = $apiResponse->getResult();

            $payment->update(['payment_token' => $order->getId()]);

            foreach ($order->getLinks() as $link) {
                if ($link->getRel() === 'approve') {
                    return $link->getHref();
                }
            }
        } else {
            $errors = $apiResponse->getResult();
            return json_encode(['error' => $errors['message']]);
        }
    }

    public function success(Payment $payment)
    {
        $orderId = $payment->payment_token;

        $paymentGateway = PaymentGateway::where('name', 'paypal')->first();
        $config = json_decode($paymentGateway->config);

        $environment = $paymentGateway->mode === 'live' ? Environment::PRODUCTION : Environment::SANDBOX;

        $client = PaypalServerSdkClientBuilder::init()
            ->clientCredentialsAuthCredentials(
                ClientCredentialsAuthCredentialsBuilder::init(
                    $config->client_id,
                    $config->client_secret
                )
            )
            ->environment($environment)
            ->build();

        $collect = [
            'id' => $orderId,
            'prefer' => 'return=minimal'
        ];

        $apiResponse = $client->getOrdersController()->captureOrder($collect);

        if ($apiResponse->isSuccess()) {
            /** @var \PaypalServerSdkLib\Models\Order $order */
            $order = $apiResponse->getResult();

            if ($order->getStatus() === 'COMPLETED') {
                $payment->orders()->update([
                    'payment_status' => PaymentStatus::PAID->value
                ]);

                $payment->update([
                    'is_paid' => true
                ]);

                return to_route('order.payment.success', $payment);
            } else {
                return to_route('order.payment.cancel', ['payment' => $payment, 'error' => 'Payment failed']);
            }
        } else {
            $errors = $apiResponse->getResult();
            return to_route('order.payment.cancel', ['payment' => $payment, 'error' => $errors['message']]);
        }
    }
}
