<?php

namespace App\Http\Controllers\Gateway\Stripe;

use App\Enums\PaymentStatus;
use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\PaymentGateway;
use Illuminate\Http\Request;
use Stripe\Checkout\Session;
use Stripe\Stripe;

class ProcessController extends Controller
{
    /**
     * Process to stripe
     *
     * @return string
     */
    public static function process($paymentGateway, Payment $payment)
    {
        $config = json_decode($paymentGateway->config);

        Stripe::setApiKey($config->secret_key);

        // default currency rate
        $defaultCurrency = generaleSetting('defaultCurrency');

        $defaultCurrencyRate = $defaultCurrency?->rate ?? 1;
        $rateFromDefault = 1 / $defaultCurrencyRate;

        $amount = round($payment->amount * $rateFromDefault, 2);
        if ($amount < 0.50) {
            return json_encode(['error' => 'Amount must be at least $0.50']);
        }

        $session = Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [
                [
                    'price_data' => [
                        'currency' => 'usd',
                        'product_data' => [
                            'name' => 'PaymentID #'.str_pad($payment->id, 6, '0', STR_PAD_LEFT),
                            'metadata' => [
                                'order_ids' => $payment->orders->pluck('id')->implode(','),
                                'amount' => $amount,
                                'total_orders' => $payment->orders->count(),
                            ],
                        ],
                        'unit_amount' => $amount * 100, // Amount in cents
                    ],
                    'quantity' => 1,
                ],
            ],
            'mode' => 'payment',
            'success_url' => route('stripe.payment.success', $payment),
            'cancel_url' => route('order.payment.cancel', $payment),
        ]);

        $payment->update([
            'payment_token' => $session->id
        ]);

        return $session->url;
    }

    public function success(Payment $payment)
    {
        $sessionId = $payment->payment_token;

        $paymentGateway = PaymentGateway::where('name', 'stripe')->first();
        $config = json_decode($paymentGateway->config);

        Stripe::setApiKey($config->secret_key);

        $session = Session::retrieve($sessionId);

        if ($session->payment_status === 'paid') {
            $payment->orders()->update([
                'payment_status' => PaymentStatus::PAID->value,
            ]);

            $payment->update([
                'is_paid' => true
            ]);

            return to_route('order.payment.success', $payment);
        } else {
            return to_route('order.payment.cancel', ['payment' => $payment, 'error' => 'Payment failed']);
        }
    }
}
