<?php

namespace App\Http\Requests;

use App\Rules\EmailRule;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;

class OrderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $token = request('access_token');
        $user = Auth::guard('sanctum')->user();

        $rules = [
            'shop_ids' => 'required|array',
            'shop_ids.*' => 'required|exists:shops,id',
            'address_id' => 'required|exists:addresses,id',
            'note' => 'nullable|string',
            'payment_method' => 'required|string',
            'coupon_code' => 'nullable|string|max:50',
        ];

        if (! $user) {
            $rules = array_merge($rules, [
                'shipping.name' => 'required|string|max:255',
                'shipping.country' => 'required|string|max:255',
                'shipping.phone_code' => 'required|string|max:255',
                'shipping.city_id' => 'required|exists:cities,id',
                'shipping.phone' => 'required|numeric|digits_between:9,16',
                'shipping.email' => ['required', 'email', 'max:255', new EmailRule],
                'shipping.area' => 'required|string|max:255',
                'shipping.address_line' => 'required|string|max:255',
                'shipping.address_type' => 'nullable|string|max:255',
            ]);

            $rules['address_id'] = 'nullable|exists:addresses,id';
        }

        return $rules;
    }

    public function messages(): array
    {
        $request = request();
        if ($request->is('api/*')) {
            $lan = $request->header('accept-language') ?? 'en';
            app()->setLocale($lan);
        }

        return [
            'shop_ids.required' => __('The shop field is required.'),
            'shop_ids.array' => __('The shop ids must be an array.'),
            'shop_ids.*.required' => __('The shop field is required.'),
            'shop_ids.*.exists' => __('The selected shop id is invalid.'),
            'address_id.required' => __('The address field is required.'),
            'address_id.exists' => __('The selected address id is invalid.'),
            'payment_method.required' => __('The payment method field is required.'),
            'shipping.name.required' => __('The name field is required'),
            'shipping.name.max' => __('The name may not be greater than 255 characters'),
            'shipping.name.string' => __('The name must be a string'),
            'shipping.country.required' => __('The country field is required'),
            'shipping.city_id.required' => __('The city field is required'),
            'shipping.city_id.exists' => __('The selected city is invalid'),
            'shipping.country.string' => __('The country must be a string'),
            'shipping.country.max' => __('The country may not be greater than 255 characters'),
            'shipping.phone.required' => __('The phone field is required.'),
            'shipping.phone.numeric' => __('The phone must be a valid phone number.'),
            'shipping.phone.digits_between' => __('The phone must be between 9 and 16 digits.'),
            'shipping.email.required' => __('The email field is required'),
            'shipping.email.email' => __('The email must be a valid email address'),
            'shipping.area.required' => __('The area field is required'),
            'shipping.area.max' => __('The area may not be greater than 255 characters'),
            'shipping.address_type.required' => __('The address type field is required'),
            'shipping.address_type.max' => __('The address type may not be greater than 255 characters'),
            'shipping.post_code.required' => __('The post code field is required'),
            'shipping.post_code.max' => __('The post code may not be greater than 255 characters'),
            'shipping.flat_no.max' => __('The flat no may not be greater than 255 characters'),
            'shipping.address_line.required' => __('The address line field is required'),
            'shipping.address_line.max' => __('The address line may not be greater than 255 characters'),
            'shipping.address_line2.max' => __('The address line 2 may not be greater than 255 characters'),
        ];
    }
}
