import axios from "axios";
import { defineStore } from "pinia";
import { useToast } from "vue-toastification";
import AddToCartDialog from "../components/AddCartPopupDialog.vue";
import RemoveCartPopupDialog from "../components/RemoveCartPopupDialog.vue";

import { useAuth } from "./AuthStore";
import { useGuestStore } from "./GuestStore";

const toast = useToast();

export const useBaskerStore = defineStore("basketStore", {
    state: () => ({
        total: 0,
        products: [],
        checkoutProducts: [],
        selectedShopIds: [],
        total_amount: 0,
        delivery_charge: 0,
        coupon_discount: 0,
        payable_amount: 0,
        order_tax_amount: 0,
        coupon_code: "",
        showOrderConfirmModal: false,
        orderPaymentCancelModal: false,
        address: null,
        buyNowShopId: null,
        buyNowProduct: null,
        isLoadingCart: false,
        city_id: null,
    }),

    getters: {
        totalAmount: (state) => {
            let total = 0;
            state.products.forEach((item) => {
                item.products.forEach((product) => {
                    let price =
                        product.discount_price > 0
                            ? product.discount_price
                            : product.price;
                    total += price * product.quantity;
                });
            });
            return total.toFixed(2);
        },

        totalCheckoutAmount: (state) => {
            let total = 0;
            state.checkoutProducts.forEach((item) => {
                item.products.forEach((product) => {
                    let price =
                        product.discount_price > 0
                            ? product.discount_price
                            : product.price;
                    total += price * product.quantity;
                });
            });
            return total.toFixed(2);
        },

        checkoutTotalItems: (state) => {
            let total = 0;
            state.checkoutProducts.forEach((item) => {
                total += item.products.length;
            });
            return total;
        },
    },

    actions: {
        /**
         * Add a product to cart.
         * @param {object} data - object containing product id, quantity, color, size, unit.
         * @param {object} product - the product to add to cart.
         * @returns {Promise}
         */
       async addToCart(data, product, showToast = true) {
            if (data.product_id) {
                this.isLoadingCart = true;
                const content = {
                    component: AddToCartDialog,
                    props: {
                        product: product,
                    },
                };

                const authStore = useAuth();

                if (!authStore.token) {
                    const guestStore = useGuestStore();
                    const carts = guestStore.getCarts();

                   await axios.post("/guest/cart/store", { ...data, carts })
                        .then((response) => {
                            this.total = response.data.data.total;
                            this.products = response.data.data.cart_items;
                            guestStore.updateCarts(response.data.data.carts);

                            toast(content, {
                                type: "default",
                                hideProgressBar: true,
                                icon: false,
                                position: "bottom-left",
                                toastClassName: "vue-toastification-alert",
                                timeout: 3000,
                            });
                            this.isLoadingCart = false;
                        })
                        .catch((error) => {
                            this.isLoadingCart = false;
                            if (error.response?.data?.data?.total) {
                                this.total = error.response.data.data.total;
                                this.products =
                                    error.response.data.data.cart_items;
                                guestStore.updateCarts(error.response.data.data.carts);
                            }
                            toast.error(error.response.data.message, {
                                position: "bottom-left",
                            });
                        });

                    return true;
                }

                axios.post("/cart/store", data, {
                        headers: {
                            Authorization: authStore.token,
                        },
                    }).then((response) => {
                    this.isLoadingCart = false;
                    if (!data.is_buy_now && showToast) {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                        toast(content, {
                            type: "default",
                            hideProgressBar: true,
                            icon: false,
                            position: "bottom-left",
                            toastClassName: "vue-toastification-alert",
                            timeout: 3000,
                        });
                    }
                }).catch((error) => {
                    this.isLoadingCart = false;
                    if (error.response.status == 401) {
                        toast.error("Please login first!", {
                            position: "bottom-left",
                        });
                        const authStore = useAuth();
                        authStore.showLoginModal();
                    } else {
                        toast.error(error.response.data.message, {
                            position: "bottom-left",
                        });
                    }
                    return error;
                });

                return true;
            }
        },

        /**
         * Fetches the cart data from the server and updates the state.
         * If the user is not logged in, it clears the cart and related state.
         */
        fetchCart() {
            const authStore = useAuth();

            if (authStore.token) {
                axios
                    .get("/carts", {
                        headers: {
                            Authorization: authStore.token,
                        },
                    })
                    .then((response) => {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                    })
                    .catch((error) => {
                        console.log(error);
                    });
            } else {
                const guestStore = useGuestStore();
                const carts = guestStore.getCarts();

                axios
                    .post("/guest/cart", { carts })
                    .then((response) => {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                        guestStore.updateCarts(response.data.data.carts);
                    })
                    .catch((error) => {
                        console.log(error);
                    });
            }
        },

        /**
         * Decrement the quantity of a given product in the cart
         * @param {object} product - the product to decrement the quantity for
         */
        decrementQuantity(product) {
            const authStore = useAuth();
            if (product) {
                const content = {
                    component: RemoveCartPopupDialog,
                    props: {
                        product: product,
                    },
                };

                if (!authStore.token) {
                    const guestStore = useGuestStore();
                    const carts = guestStore.getCarts();

                    axios
                        .post("/guest/cart/decrement", {
                            product_id: product.id,
                            carts,
                        })
                        .then((response) => {
                            this.total = response.data.data.total;
                            this.products = response.data.data.cart_items;
                            guestStore.updateCarts(response.data.data.carts);
                            this.fetchCheckoutProducts();

                            if (
                                response.data.message ==
                                "product removed from cart"
                            ) {
                                const shopIds = this.products.map(
                                    (shop) => shop.shop_id
                                );
                                this.selectedShopIds =
                                    this.selectedShopIds.filter((id) =>
                                        shopIds.includes(id)
                                    );
                                // const exists = shopIds.some((id) => selectedShopIds.includes(id));

                                if (this.products.length === 0) {
                                    this.selectedShopIds = [];
                                    this.checkoutProducts = [];
                                    this.total_amount = 0;
                                    this.delivery_charge = 0;
                                    this.coupon_discount = 0;
                                    this.payable_amount = 0;
                                }

                                toast(content, {
                                    type: "default",
                                    hideProgressBar: true,
                                    icon: false,
                                    position: "bottom-left",
                                    toastClassName: "vue-toastification-alert",
                                    timeout: 3000,
                                });
                            }
                        })
                        .catch((error) => {
                            if (error.response?.data?.data?.total) {
                                this.total = error.response.data.data.total;
                                this.products =
                                    error.response.data.data.cart_items;
                                guestStore.updateCarts(error.response.data.data.carts);
                            }

                            toast.error(error.response.data.message, {
                                position: "bottom-left",
                            });
                        });

                    return;
                }

                axios
                    .post(
                        "/cart/decrement",
                        { product_id: product.id },
                        {
                            headers: {
                                Authorization: authStore.token,
                            },
                        }
                    )
                    .then((response) => {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                        this.fetchCheckoutProducts();

                        if (
                            response.data.message == "product removed from cart"
                        ) {
                            const shopIds = this.products.map(
                                (shop) => shop.shop_id
                            );
                            this.selectedShopIds = this.selectedShopIds.filter(
                                (id) => shopIds.includes(id)
                            );
                            // const exists = shopIds.some((id) => selectedShopIds.includes(id));

                            if (this.products.length === 0) {
                                this.selectedShopIds = [];
                                this.checkoutProducts = [];
                                this.total_amount = 0;
                                this.delivery_charge = 0;
                                this.coupon_discount = 0;
                                this.payable_amount = 0;
                            }

                            toast(content, {
                                type: "default",
                                hideProgressBar: true,
                                icon: false,
                                position: "bottom-left",
                                toastClassName: "vue-toastification-alert",
                                timeout: 3000,
                            });
                        }
                    });
            }
        },

        /**
         * Increment the quantity of the given product in the cart
         * @param {object} product - the product to increment the quantity for
         */
        incrementQuantity(product) {
            const authStore = useAuth();
            if (product) {
                if (!authStore.token) {
                    const guestStore = useGuestStore();
                    const carts = guestStore.getCarts();

                    axios
                        .post("/guest/cart/increment", {
                            product_id: product.id,
                            carts,
                        })
                        .then((response) => {
                            this.total = response.data.data.total;
                            this.products = response.data.data.cart_items;
                            guestStore.updateCarts(response.data.data.carts);
                            this.fetchCheckoutProducts();
                        })
                        .catch((error) => {
                            if (error.response?.data?.data?.total) {
                                this.total = error.response.data.data.total;
                                this.products =
                                    error.response.data.data.cart_items;
                                guestStore.updateCarts(error.response.data.data.carts);
                            }

                            toast.error(error.response.data.message, {
                                position: "bottom-left",
                            });
                        });

                    return;
                }

                axios
                    .post("/cart/increment",
                        { product_id: product.id },
                        {
                            headers: {
                                Authorization: authStore.token,
                            },
                        }
                    )
                    .then((response) => {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                        this.fetchCheckoutProducts();
                    })
                    .catch((error) => {
                        toast.error(error.response.data.message, {
                            position: "bottom-left",
                        });
                    });
            }
        },

        /**
         * Remove the given product from the cart
         * @param {object} product - the product to remove from the cart
         */
        removeFromBasket(product) {
            const authStore = useAuth();
            if (product) {
                if (!authStore.token) {
                    const guestStore = useGuestStore();
                    const carts = guestStore.getCarts();

                    axios
                        .post("/guest/cart/delete", {
                            product_id: product.id,
                            carts,
                        })
                        .then((response) => {
                            this.total = response.data.data.total;
                            this.products = response.data.data.cart_items;
                            guestStore.updateCarts(response.data.data.carts);
                            this.fetchCheckoutProducts();
                        })
                        .catch((error) => {
                            if (error.response?.data?.data?.total) {
                                this.total = error.response.data.data.total;
                                this.products = error.response.data.data.cart_items;
                                guestStore.updateCarts(error.response.data.data.carts);
                            }
                        });

                    return;
                }

                axios.post("/cart/delete",
                        { product_id: product.id },
                        {
                            headers: {
                                Authorization: authStore.token,
                            },
                        }
                    ).then((response) => {
                        this.total = response.data.data.total;
                        this.products = response.data.data.cart_items;
                        this.fetchCheckoutProducts();
                    });
            }
        },

        /**
         * Select or deselect the given shop for checkout
         * @param {number} shop - the shop to select or deselect
         */
        selectCartItemsForCheckout(shop) {
            if (!this.selectedShopIds.includes(shop)) {
                this.selectedShopIds.push(shop);
            } else {
                this.selectedShopIds = this.selectedShopIds.filter(
                    (item) => item !== shop
                );
            }
            this.fetchCheckoutProducts();
        },

        /**
         * Fetches the checkout products for the currently selected shops and updates
         * the checkout-related state, including total amount, delivery charge, coupon
         * discount, and payable amount. If the checkout products are empty, it clears
         * the selected shop IDs. Uses the auth token for authorization.
         */
        fetchCheckoutProducts() {
            const authStore = useAuth();

            if (authStore.token) {
                axios.post("/cart/checkout",{
                    shop_ids: this.selectedShopIds,
                    address_id: this.address?.id
                },
                {
                    headers: {
                        Authorization: authStore.token,
                    },
                }).then((response) => {
                    this.checkoutProducts = response.data.data.checkout_items;
                    this.total_amount = response.data.data.checkout.total_amount;
                    this.delivery_charge = response.data.data.checkout.delivery_charge;
                    this.coupon_discount = response.data.data.checkout.coupon_discount;
                    this.payable_amount = response.data.data.checkout.payable_amount;
                    this.order_tax_amount = response.data.data.checkout.order_tax_amount;
                    if (this.checkoutProducts.length === 0) {
                        this.selectedShopIds = [];
                    }
                }).catch((error) => {
                    toast.error(error.response.data.message);
                });
            } else {
                const guestStore = useGuestStore();

                axios.post("/guest/cart/checkout", {
                    shop_ids: this.selectedShopIds,
                    carts: guestStore.getCarts(),
                    collected_coupons: guestStore.getColllectedCoupons(),
                    city_id: this.city_id
                }).then((response) => {
                    this.total = response.data.data.total;
                    this.products = response.data.data.cart_items;
                    this.checkoutProducts =
                        response.data.data.checkout_items;
                    this.total_amount =
                        response.data.data.checkout.total_amount;
                    this.delivery_charge =
                        response.data.data.checkout.delivery_charge;
                    this.coupon_discount =
                        response.data.data.checkout.coupon_discount;
                    this.payable_amount =
                        response.data.data.checkout.payable_amount;
                    if (this.checkoutProducts.length === 0) {
                        this.selectedShopIds = [];
                    }
                    guestStore.updateCollectedCoupons(
                        response.data.data.collected_coupons
                    );
                    guestStore.updateCarts(response.data.data.carts);
                }).catch((error) => {
                    toast.error(error.response.data.message);
                });
            }
        },

        checkShopIsSelected(shopId) {
            return this.selectedShopIds.includes(shopId);
        },
    },

    persist: true,
});
